import xml.etree.ElementTree as ET
from docplex.cp.model import *  

from ConstraintBuilder import ConstraintBuilder
from SlotReqHandler import SlotReqHandler
from OrientamentoHandler import OrientamentoHandler
from InsegnamentoHandler import InsegnamentoHandler
from LocaliHandler import LocaliHandler
from SolHandler import SolutionHandler
from RulePenalty import getAvgPenaltyDocente, getAvgPenaltyOrientamenti, getAvgPenaltyStudente
from auxiliary import AuxiliaryStruct
from dbAPI import dbAPI
import os
from datetime import datetime
from templates import *
from Slot import *
from typing import List
from log import logger

from data.tables import list_ID_INC_magistrali
from Parameter import Parameter



class Builder:
    def __init__(self):
        self.PARAM = Parameter()
        
        # auxiliary structure handle
        self.dbAPI:dbAPI = dbAPI()
        self.AUX = AuxiliaryStruct()
        self.log:logger = logger()
        
        self.SH:SlotReqHandler = SlotReqHandler()
        self.OH:OrientamentoHandler = OrientamentoHandler()
        self.IH:InsegnamentoHandler = InsegnamentoHandler()
        self.LH:LocaliHandler = LocaliHandler()
        
        self.docentiSet = set()
        
        # controls
        self.foldersLoaded = False
        self.platform = "portatile"

        self.model:CpoModel = CpoModel("model1")


    def loadFileInsegnamenti(self, fileInsegnamenti):
        tree = ET.parse(fileInsegnamenti)
        root = tree.getroot()

        for TemplateOrario in root:
            insegnamento = MetaInsegnamento(str(TemplateOrario[0].text),str(TemplateOrario[1].text),int(TemplateOrario[2].text))
            
            if int(insegnamento.ID_INC) in [-5]:#259400,259401]:#,261945]:#, 262243]:
                # continue
                pass
            
            if self.PARAM.usePianoAllocazioneAsBase:
                # carico da xml solo gli Insegnamenti allocabili in modo dinamico
                if int(insegnamento.ID_INC) in self.PARAM.listID_INCToModify:
                    idIns = self.AUX.add_metaInsegnamentoToList(insegnamento)
                    self.SH.loadInsegnamento(TemplateOrario, idIns)
            else:
                # modalità normale
                idIns = self.AUX.add_metaInsegnamentoToList(insegnamento)
                self.SH.loadInsegnamento(TemplateOrario, idIns)
                
                       
    def loadFoldersXml(self, folderList:List[str]):
        '''Chiamata una volta sola, carica tutte le folder con i file XML contenenti Insegnamenti e Docenti.
        
        Params:
            folderList (list): lista delle cartelle da caricare'''
        
        if self.foldersLoaded:
            self.log.error_log("loadFoldersXml(): non può più essere chiamato")
            exit(-1)
        self.foldersLoaded = True  # una volta generate le struct aux non si può più chiamare
        
        self.log.info_log("loading Orario: use " + str(self.PARAM.nFilesInsegnamenti) + " ")
        i = 0
        listFileDone = list()
        # start loading Docenti
        for folder in folderList:
            for file in os.listdir(folder):
                try:
                    self.loadFileDocenti(folder + "\\" + file)
                    listFileDone.append(folder + "\\" + file)
                    self.AUX.add_fileToListFileDone(file)
                except Exception as e:
                    self.log.error_log("builder.loadFileDocenti(): error on parsing file " + folder + "\\" + file)
                    self.log.error_log(str(e))
                    exit(-1)
                finally:
                    if self.platform == "portatile" and not i < self.PARAM.nFilesInsegnamenti:
                        break
                    i+=1
        
        # creazione AUX map docenti
        self.AUX.createMap_strDocenti_to_IdDocenti(self.docentiSet)
        
        # start loading Insegnamenti
        for file in listFileDone:
            try:
                self.loadFileInsegnamenti(file)
            except Exception as e:
                self.log.error_log("builder.loadFileInsegnamenti(): error on parsing file " + file)
                self.log.error_log(str(e))
                exit(-1)  
        
        if self.PARAM.usePianoAllocazioneAsBase:
            # in questa modalità a questo punto devo caricare da db tutti gli ID_INC presenti nel Piano Allocazione ma non facenti parte degli
            # Insegnamenti modificabili -> saranno caricati come interamente composti da SlotScelti
            listID_INC = self.dbAPI.get_ID_INCFromPianoAllocazione(self.PARAM.pianoAllocazioneBase)
            if len(listID_INC) == 0:
                self.log.error_log("builder.loadFileInsegnamenti(): errore precaricamento piano allocazione base: " + str(self.PARAM.pianoAllocazioneBase))
                return
            
            for ID_INC in listID_INC:
                # questi li prendo dai file xml
                if int(ID_INC[0]) in self.PARAM.listID_INCToModify:
                    continue
                
                insegnamentoDataDb = self.dbAPI.get_Insegnamento(int(ID_INC[0]))
                insegnamento = MetaInsegnamento(insegnamentoDataDb[0][4], str(ID_INC[0]), int(insegnamentoDataDb[0][2]))
                
                idIns = self.AUX.add_metaInsegnamentoToList(insegnamento)
                self.SH.loadInsegnamentoFromDb(self.PARAM.pianoAllocazioneBase, int(ID_INC[0]), idIns)                
                
        self.AUX.end_metaInsegnamentoToList() 
        self.AUX.end_pianoAllocazione()
        
        # check all Insegnamenti loaded
        listInsDb = self.dbAPI.get_Insegnamenti()
        for insDb in listInsDb:
            found:bool = False
            for insXml in self.AUX.list_metaInsegnamenti:
                if insXml.ID_INC == str(insDb[0]):
                    found = True
                    break
            if not found:
                self.log.error_log("Insegnamento non caricato: " + str(insDb[0]) + " - " + insDb[4] + " - " + insDb[7])
        
        
    def loadFileDocenti(self, fileDocenti:str):
        '''Crea la struttura self.docentiSet:Set[str]. Serve per il mapping Docente:str -> id:int'''
        tree = ET.parse(fileDocenti)
        root = tree.getroot()
        
        for TemplateOrario in root:
            for child in TemplateOrario:
                # per ogni insegnamento
                if child.tag == "SlotGenerico" or child.tag == "SlotScelto":
                    for cc in child:
                        if cc.tag == "Docente":
                            if cc.text != "Docente":
                                self.docentiSet.add(cc.text) 
        

    
    def loadModel(self):
        '''Carica tutti i dati del modello'''
        # 1) xml insegnamenti
        self.log.info_log("\nbuilder.loadModel(): load Insegnamenti and Docenti")
        self.loadFoldersXml(self.PARAM.folderInsegnamentiXml)
        # 2) load aule
        self.log.info_log("\nbuilder.loadModel(): load Locali")
        self.LH.loadXmlLocaliAllocati(self.PARAM.fileXmlAule)

    def buildModel(self):
        '''Crea le strutture, i vincoli e le equazioni per il modello'''
        # create AUX structures
        self.AUX.createMap_strSlotId_to_IdSlot()
        self.AUX.createList_docentiInSlot()            
        self.AUX.createList_slotInDocente()
        self.AUX.createList_slotInInsegnamento()
                
        # Caricamento stuff dal db, mi servono già le strutture AUX
        self.OH.loadOrientamenti()
        self.IH.loadInsegnamenti()
        
        # stuff con xml
        self.AUX.end_listVincoliInsegnamenti()
        self.AUX.end_listOperatoriInsegnamenti()
        
        # create AUX structures per cui servono Orientamenti ed Insegnamenti
        self.AUX.createList_InsegnamentoInOrientamento()
        self.AUX.createList_slotIdInOrientamento()
        self.AUX.createList_InsegnamentiParallelizzabili()
                
        # self.AUX.debug()
        
        self.log.info_log("builder.buildModel() dimensioni: num_slot: " + str(len(self.AUX.map_strSlotId_to_idSlot)))        

        # for the day
        self.X_d = self.model.integer_var_list(self.AUX.get_nSlotId(), 0, self.AUX.get_NUM_DAY()-1, "X_d")
        # for the hour
        self.X_h = self.model.integer_var_list(self.AUX.get_nSlotId(), 0, self.AUX.NUM_SLOT_PER_DAY-1, "X_h")
    
        self.CB = ConstraintBuilder(self.model, self.X_d, self.X_h)
        
    
    def buildConstraint(self):
        # CB constraints
        self.log.info_log("\nbuilder.buildConstraint.CB(): start")  
        now = datetime.now()
        current_time = now.strftime("%H:%M:%S")
        self.log.info_log("Start Time = " + current_time)
        
        self.X_penalties,self.nVar_X_penalties, self.X_penaltiesStud, self.nVar_X_penaltiesStud, self.X_penaltiesDoc, self.nVar_X_penaltiesDoc = self.CB.addConstraint()
        
        self.log.info_log("builder.buildConstraint.CB(): end\n")  
        now = datetime.now()
        current_time = now.strftime("%H:%M:%S")
        self.log.info_log("End Time = " + current_time)
        self.model.write_information()
      
        
    def solve(self) -> bool:
        '''Solva il modello'''
        
        # build dei constraints rispettando il tipo di target
        self.buildConstraint()
        
        # pulizia del db prima di salvare le nuove sol
        pianiAllocToDel = list(map(lambda nameTuple: nameTuple[0], self.dbAPI.get_groupPianiAllocazione(self.PARAM.nomeSolutions)))
        try:
            for pianoAlloc in pianiAllocToDel:
                self.log.info_log("builder.solve(): delete pianoAllocazione: " + pianoAlloc)
                self.dbAPI.delete_pianoAllocazione(pianoAlloc, False)
            self.dbAPI.commit()
        except Exception as e:
            self.log.error_log("builder.solve(): errore pulizia db da vecchi pianiAllocazione: " + str(e))
            self.dbAPI.rollback()
        


        len_ex = lambda lst: len(lst) if lst is not None else 0

        # configurazione del modello in funzione della logica di risoluzione stabilita
        if self.PARAM.modelConfig == Parameter.OptimizeComponent.Nessuno:
            pass        
        # set la funzione obiettivo desiderata
        elif self.PARAM.modelConfig == Parameter.OptimizeComponent.Orientamento|Parameter.OptimizeComponent.Studenti:
            self.model.minimize(self.model.sum(self.X_penalties[i] for i in range(len_ex(self.X_penalties))) + self.model.sum(self.X_penaltiesStud[i] for i in range(len_ex(self.X_penaltiesStud))))
        elif self.PARAM.modelConfig == Parameter.OptimizeComponent.Orientamento|Parameter.OptimizeComponent.Docenti:
            self.model.minimize(self.model.sum(self.X_penalties[i] for i in range(len_ex(self.X_penalties))) + self.model.sum(self.X_penaltiesDoc[i] for i in range(len_ex(self.X_penaltiesDoc))))
        elif self.PARAM.modelConfig == Parameter.OptimizeComponent.Docenti|Parameter.OptimizeComponent.Studenti:
            self.model.minimize(self.model.sum(self.X_penaltiesStud[i] for i in range(len_ex(self.X_penaltiesStud))) + self.model.sum(self.X_penaltiesDoc[i] for i in range(len_ex(self.X_penaltiesDoc))))
        elif self.PARAM.modelConfig == Parameter.OptimizeComponent.Orientamento|Parameter.OptimizeComponent.Docenti|Parameter.OptimizeComponent.Studenti:
            self.model.minimize(self.model.sum(self.X_penalties[i] for i in range(len_ex(self.X_penalties))) + self.model.sum(self.X_penaltiesStud[i] for i in range(len_ex(self.X_penaltiesStud))) + self.model.sum(self.X_penaltiesDoc[i] for i in range(len_ex(self.X_penaltiesDoc))))
        elif self.PARAM.modelConfig == Parameter.OptimizeComponent.Orientamento:
            self.model.minimize(self.model.sum(self.X_penalties[i] for i in range(len_ex(self.X_penalties))))
        elif self.PARAM.modelConfig == Parameter.OptimizeComponent.Docenti:
            self.model.minimize(self.model.sum(self.X_penaltiesDoc[i] for i in range(len_ex(self.X_penaltiesDoc))))
        elif self.PARAM.modelConfig == Parameter.OptimizeComponent.Studenti:
            self.model.minimize(self.model.sum(self.X_penaltiesStud[i] for i in range(len_ex(self.X_penaltiesStud))))
            
                                          
        if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Orientamento:
            self.log.info_log("builder.solve(): limite penalità per gli Orientamenti: " + str(self.nVar_X_penalties*getAvgPenaltyOrientamenti()))
            # limite penalità media tra tutti gli Insegnamenti
            self.model.add(self.model.sum(self.X_penalties[i] for i in range(len_ex(self.X_penalties)))
                           <= self.nVar_X_penalties*getAvgPenaltyOrientamenti())

        if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Studenti:
            self.log.info_log("builder.solve(): limite penalità per gli Studenti: " + str(self.nVar_X_penaltiesStud*getAvgPenaltyStudente()))
            # limite penalità media tra tutte le penalità
            self.model.add(self.model.sum(self.X_penaltiesStud[i] for i in range(len_ex(self.X_penaltiesStud))) 
                           <= self.nVar_X_penaltiesStud*getAvgPenaltyStudente())
            
        if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Docenti:
            self.log.info_log("builder.solve(): limite penalità per i Docenti: " + str(self.nVar_X_penaltiesDoc*getAvgPenaltyDocente()))
            # limite penalità media tra tutti i Docenti
            self.model.add(self.model.sum(self.X_penaltiesDoc[i] for i in range(len_ex(self.X_penaltiesDoc))) 
                           <= self.nVar_X_penaltiesDoc*getAvgPenaltyDocente())

            for docId,listPenalties in self.CB.RDH.map_penaltiesInDocente.items():
                if len(listPenalties) == 0:
                    continue # for safe (non dovrebbe servire)
                self.log.info_log("builder.solve(): per il docente " + self.AUX.map_idDocenti_to_strDocenti[docId] + " ho una penalità massima di: " + str(getAvgPenaltyDocente(self.AUX.map_idDocenti_to_strDocenti[docId])))
                self.model.add(self.model.sum(self.X_penaltiesDoc[i] for i in listPenalties) <= len(listPenalties)*getAvgPenaltyDocente(self.AUX.map_idDocenti_to_strDocenti[docId]))
                            
        # add solution recorder
        self.model.add_solver_listener(SolutionHandler())
        
        # set the time limit
        if self.PARAM.timeLimitWork == -1:
            if self.PARAM.nWorkers == -1:
                self.mSols:CpoSolveResult = self.model.start_search(LogPeriod=self.PARAM.CPLEX_LogPeriod,SolutionLimit=None)   
            else:
                self.mSols:CpoSolveResult = self.model.start_search(Workers=self.PARAM.nWorkers,
                    LogPeriod=self.PARAM.CPLEX_LogPeriod,SolutionLimit=None)
        else:
            if self.PARAM.nWorkers == -1:
                self.mSols:CpoSolveResult = self.model.start_search(TimeLimit=self.PARAM.timeLimitWork,
                    LogPeriod=self.PARAM.CPLEX_LogPeriod,SolutionLimit=None)
            else:
                self.mSols:CpoSolveResult = self.model.start_search(TimeLimit=self.PARAM.timeLimitWork, 
                    Workers=self.PARAM.nWorkers, LogPeriod=self.PARAM.CPLEX_LogPeriod,SolutionLimit=None)
                
        self.log.info_log("builder.solve(): start")
        now = datetime.now()
        current_time = now.strftime("%H:%M:%S")
        self.log.info_log("Start Time = " + current_time)          
                
        for s in self.mSols:
            self.mSolution = s
            
            if not self.mSolution:
                self.log.error_log("builder.solve(): NO SOLUTION FOUND")
                return False
            if self.mSolution and self.PARAM.modelConfig == Parameter.OptimizeComponent.Nessuno and self.PARAM.additionalModelConfig == Parameter.OptimizeComponent.Nessuno:
                break
        
        now = datetime.now()
        current_time = now.strftime("%H:%M:%S")
        self.log.info_log("Found sol at time = " + current_time + "\n")
        self.model.write_information()            
        return True
    
    
    